﻿using System.Collections.Generic;

namespace Hims.Api.Controllers
{
    using System;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels.Common;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Shared.UserModels.ProgressReport.Diet;
    using Utilities;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/progress-report-diet")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ProgressReportDietController : BaseController
    {
        /// <summary>
        /// The progress report vitals services.
        /// </summary>
        private readonly IProgressReportDietService service;

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IPatientService patientServices;

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IProviderService providerServices;

        /// <summary>
        /// the audit service
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public ProgressReportDietController(IProgressReportDietService service, IPatientService patientServices, IProviderService providerServices, IAuditLogService auditLogService)
        {
            this.service = service;
            this.patientServices = patientServices;
            this.providerServices = providerServices;
            this.auditLogServices = auditLogService;
        }

        /// <summary>
        /// Fetches the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertAsync([FromBody] InsertModel model,[FromHeader] LocationHeader header)
        {
            model = (InsertModel)EmptyFilter.Handler(model);
            try
            {
                var response = await this.service.InsertAsync(model);

                var patientFullName = await this.patientServices.FindPatientByAdmissionId(model.AdmissionId);
                var providerName = await this.providerServices.FindProviderByAdmissionId(model.AdmissionId);
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.ProgressReport,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" {model.ModifiedByName} has added Progress report for Patient {patientFullName} on {DateTime.UtcNow.AddMinutes(330)} Prescribed by Doctor {providerName}.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception e)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = e.Message
                });
            }
        }

        /// <summary>
        /// Fetches the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] FilterModel model)
        {

            var response = await this.service.FetchAsync(model);
            return Ok(new GenericResponse
            {
                Status = GenericStatus.Success,
                Data = response
            });

        }

        /// <summary>
        /// Fetches the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("stop-diet")]
        public async Task<ActionResult> StopDietAsync([FromBody] StopModel model, [FromHeader] LocationHeader header)
        {
            model = (StopModel)EmptyFilter.Handler(model);

            var patientFullName = await this.patientServices.FindPatientByAdmissionId(model.AdmissionId);
            var providerName = await this.providerServices.FindProviderByAdmissionId(model.AdmissionId);
            var response = await this.service.StopDietAsync(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.By,
                LogTypeId = (int)LogTypes.ProgressReport,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" <b>{model.CreatedBy}</b> has Stopped Diet for Patient <b>{patientFullName}</b> on {DateTime.UtcNow.AddMinutes(330)} Prescribed by Doctor <b>{providerName}</b>.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return Ok(response);
        }
    }
}
